/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.procedure2;

import com.google.common.base.Preconditions;
import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ProcedureInfo;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.classification.InterfaceStability;
import org.apache.hadoop.hbase.exceptions.IllegalArgumentIOException;
import org.apache.hadoop.hbase.procedure2.Procedure;
import org.apache.hadoop.hbase.procedure2.ProcedureRunnableSet;
import org.apache.hadoop.hbase.procedure2.ProcedureSimpleRunQueue;
import org.apache.hadoop.hbase.procedure2.ProcedureYieldException;
import org.apache.hadoop.hbase.procedure2.RemoteProcedureException;
import org.apache.hadoop.hbase.procedure2.RootProcedureState;
import org.apache.hadoop.hbase.procedure2.store.ProcedureStore;
import org.apache.hadoop.hbase.procedure2.util.StringUtils;
import org.apache.hadoop.hbase.procedure2.util.TimeoutBlockingQueue;
import org.apache.hadoop.hbase.protobuf.generated.ProcedureProtos;
import org.apache.hadoop.hbase.security.User;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.ForeignExceptionUtil;
import org.apache.hadoop.hbase.util.NonceKey;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class ProcedureExecutor<TEnvironment> {
    private static final Log LOG = LogFactory.getLog(ProcedureExecutor.class);
    Testing testing = null;
    private final ConcurrentHashMap<Long, ProcedureInfo> completed = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, RootProcedureState> rollbackStack = new ConcurrentHashMap();
    private final ConcurrentHashMap<Long, Procedure> procedures = new ConcurrentHashMap();
    private ConcurrentHashMap<NonceKey, Long> nonceKeysToProcIdsMap = new ConcurrentHashMap();
    private final TimeoutBlockingQueue<Procedure> waitingTimeout = new TimeoutBlockingQueue<Procedure>(new ProcedureTimeoutRetriever());
    private final ProcedureRunnableSet runnables;
    private final ReentrantLock submitLock = new ReentrantLock();
    private final AtomicLong lastProcId = new AtomicLong(-1L);
    private final CopyOnWriteArrayList<ProcedureExecutorListener> listeners = new CopyOnWriteArrayList();
    private final AtomicInteger activeExecutorCount = new AtomicInteger(0);
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final TEnvironment environment;
    private final ProcedureStore store;
    private final Configuration conf;
    private Thread[] threads;

    public ProcedureExecutor(Configuration conf, TEnvironment environment, ProcedureStore store) {
        this(conf, environment, store, new ProcedureSimpleRunQueue());
    }

    public ProcedureExecutor(Configuration conf, TEnvironment environment, ProcedureStore store, ProcedureRunnableSet runqueue) {
        this.environment = environment;
        this.runnables = runqueue;
        this.store = store;
        this.conf = conf;
    }

    private void load(final boolean abortOnCorruption) throws IOException {
        Preconditions.checkArgument((boolean)this.completed.isEmpty());
        Preconditions.checkArgument((boolean)this.rollbackStack.isEmpty());
        Preconditions.checkArgument((boolean)this.procedures.isEmpty());
        Preconditions.checkArgument((boolean)this.waitingTimeout.isEmpty());
        Preconditions.checkArgument((this.runnables.size() == 0 ? 1 : 0) != 0);
        this.store.load(new ProcedureStore.ProcedureLoader(){

            @Override
            public void setMaxProcId(long maxProcId) {
                assert (ProcedureExecutor.this.lastProcId.get() < 0L) : "expected only one call to setMaxProcId()";
                LOG.debug((Object)("load procedures maxProcId=" + maxProcId));
                ProcedureExecutor.this.lastProcId.set(maxProcId);
            }

            @Override
            public void load(ProcedureStore.ProcedureIterator procIter) throws IOException {
                ProcedureExecutor.this.loadProcedures(procIter, abortOnCorruption);
            }

            @Override
            public void handleCorrupted(ProcedureStore.ProcedureIterator procIter) throws IOException {
                int corruptedCount = 0;
                while (procIter.hasNext()) {
                    ProcedureInfo proc = procIter.nextAsProcedureInfo();
                    LOG.error((Object)("corrupted procedure: " + proc));
                    ++corruptedCount;
                }
                if (abortOnCorruption && corruptedCount > 0) {
                    throw new IOException("found " + corruptedCount + " procedures on replay");
                }
            }
        });
    }

    private void loadProcedures(ProcedureStore.ProcedureIterator procIter, boolean abortOnCorruption) throws IOException {
        boolean isDebugEnabled = LOG.isDebugEnabled();
        int runnablesCount = 0;
        while (procIter.hasNext()) {
            long procId;
            NonceKey nonceKey;
            Object proc;
            if (procIter.isNextCompleted()) {
                proc = procIter.nextAsProcedureInfo();
                nonceKey = proc.getNonceKey();
                procId = proc.getProcId();
                this.completed.put(proc.getProcId(), (ProcedureInfo)proc);
                if (isDebugEnabled) {
                    LOG.debug((Object)("The procedure is completed: " + proc));
                }
            } else {
                proc = procIter.nextAsProcedure();
                nonceKey = ((Procedure)proc).getNonceKey();
                procId = ((Procedure)proc).getProcId();
                if (!((Procedure)proc).hasParent()) {
                    assert (!((Procedure)proc).isFinished()) : "unexpected finished procedure";
                    this.rollbackStack.put(((Procedure)proc).getProcId(), new RootProcedureState());
                }
                ((Procedure)proc).beforeReplay(this.getEnvironment());
                this.procedures.put(((Procedure)proc).getProcId(), (Procedure)proc);
                if (((Procedure)proc).getState() == ProcedureProtos.ProcedureState.RUNNABLE) {
                    ++runnablesCount;
                }
            }
            if (nonceKey == null) continue;
            this.nonceKeysToProcIdsMap.put(nonceKey, procId);
        }
        ArrayList<Procedure> runnableList = new ArrayList<Procedure>(runnablesCount);
        HashSet<Procedure> waitingSet = null;
        procIter.reset();
        while (procIter.hasNext()) {
            Procedure parent;
            Long rootProcId;
            if (procIter.isNextCompleted()) {
                procIter.skipNext();
                continue;
            }
            Procedure proc = procIter.nextAsProcedure();
            assert (!proc.isFinished() || proc.hasParent()) : "unexpected completed proc=" + proc;
            if (isDebugEnabled) {
                LOG.debug((Object)String.format("Loading procedure state=%s isFailed=%s: %s", proc.getState(), proc.hasException(), proc));
            }
            if ((rootProcId = this.getRootProcedureId(proc)) == null) {
                this.runnables.addBack(proc);
                continue;
            }
            if (proc.hasParent() && !proc.isFinished() && (parent = this.procedures.get(proc.getParentProcId())) != null) {
                parent.incChildrenLatch();
            }
            RootProcedureState procStack = this.rollbackStack.get(rootProcId);
            procStack.loadStack(proc);
            switch (proc.getState()) {
                case RUNNABLE: {
                    runnableList.add(proc);
                    break;
                }
                case WAITING_TIMEOUT: {
                    if (waitingSet == null) {
                        waitingSet = new HashSet<Procedure>();
                    }
                    waitingSet.add(proc);
                    break;
                }
                case FINISHED: {
                    if (proc.hasException()) {
                        this.runnables.addBack(proc);
                        break;
                    }
                }
                case ROLLEDBACK: 
                case INITIALIZING: {
                    String msg = "Unexpected " + proc.getState() + " state for " + proc;
                    LOG.error((Object)msg);
                    throw new UnsupportedOperationException(msg);
                }
            }
        }
        int corruptedCount = 0;
        Iterator<Map.Entry<Long, RootProcedureState>> itStack = this.rollbackStack.entrySet().iterator();
        while (itStack.hasNext()) {
            Map.Entry<Long, RootProcedureState> entry = itStack.next();
            RootProcedureState procStack = entry.getValue();
            if (procStack.isValid()) continue;
            for (Procedure proc : procStack.getSubprocedures()) {
                LOG.error((Object)("corrupted procedure: " + proc));
                this.procedures.remove(proc.getProcId());
                runnableList.remove(proc);
                if (waitingSet != null) {
                    waitingSet.remove(proc);
                }
                ++corruptedCount;
            }
            itStack.remove();
        }
        if (abortOnCorruption && corruptedCount > 0) {
            throw new IOException("found " + corruptedCount + " procedures on replay");
        }
        if (!runnableList.isEmpty()) {
            for (int i = runnableList.size() - 1; i >= 0; --i) {
                Procedure proc = (Procedure)runnableList.get(i);
                if (!proc.hasParent()) {
                    this.sendProcedureLoadedNotification(proc.getProcId());
                }
                if (proc.wasExecuted()) {
                    this.runnables.addFront(proc);
                    continue;
                }
                this.runnables.addBack(proc);
            }
        }
    }

    public void start(int numThreads, boolean abortOnCorruption) throws IOException {
        int i;
        if (this.running.getAndSet(true)) {
            LOG.warn((Object)"Already running");
            return;
        }
        this.threads = new Thread[numThreads + 1];
        LOG.info((Object)("Starting procedure executor threads=" + this.threads.length));
        for (i = 0; i < numThreads; ++i) {
            this.threads[i] = new Thread("ProcedureExecutor-" + i){

                @Override
                public void run() {
                    ProcedureExecutor.this.execLoop();
                }
            };
        }
        this.threads[numThreads] = new Thread("ProcedureExecutorTimeout"){

            @Override
            public void run() {
                ProcedureExecutor.this.timeoutLoop();
            }
        };
        this.store.recoverLease();
        this.load(abortOnCorruption);
        for (i = 0; i < this.threads.length; ++i) {
            this.threads[i].start();
        }
        this.waitingTimeout.add(new CompletedProcedureCleaner(this.conf, this.store, this.completed, this.nonceKeysToProcIdsMap));
    }

    public void stop() {
        if (!this.running.getAndSet(false)) {
            return;
        }
        LOG.info((Object)"Stopping the procedure executor");
        this.runnables.signalAll();
        this.waitingTimeout.signalAll();
    }

    public void join() {
        boolean interrupted = false;
        for (int i = 0; i < this.threads.length; ++i) {
            try {
                this.threads[i].join();
                continue;
            }
            catch (InterruptedException ex) {
                interrupted = true;
            }
        }
        if (interrupted) {
            Thread.currentThread().interrupt();
        }
        this.completed.clear();
        this.rollbackStack.clear();
        this.procedures.clear();
        this.nonceKeysToProcIdsMap.clear();
        this.waitingTimeout.clear();
        this.runnables.clear();
        this.lastProcId.set(-1L);
    }

    public boolean isRunning() {
        return this.running.get();
    }

    public int getNumThreads() {
        return this.threads == null ? 0 : this.threads.length - 1;
    }

    public int getActiveExecutorCount() {
        return this.activeExecutorCount.get();
    }

    public TEnvironment getEnvironment() {
        return this.environment;
    }

    public ProcedureStore getStore() {
        return this.store;
    }

    public void registerListener(ProcedureExecutorListener listener) {
        this.listeners.add(listener);
    }

    public boolean unregisterListener(ProcedureExecutorListener listener) {
        return this.listeners.remove(listener);
    }

    public List<ProcedureInfo> listProcedures() {
        ArrayList<ProcedureInfo> procedureLists = new ArrayList<ProcedureInfo>(this.procedures.size() + this.completed.size());
        for (Map.Entry<Long, Procedure> entry : this.procedures.entrySet()) {
            procedureLists.add(Procedure.createProcedureInfo(entry.getValue(), null));
        }
        for (Map.Entry<Long, Procedure> entry : this.completed.entrySet()) {
            procedureLists.add((ProcedureInfo)entry.getValue());
        }
        return procedureLists;
    }

    public NonceKey createNonceKey(long nonceGroup, long nonce) {
        return nonce == 0L ? null : new NonceKey(nonceGroup, nonce);
    }

    public long registerNonce(NonceKey nonceKey) {
        long newProcId;
        if (nonceKey == null) {
            return -1L;
        }
        Long oldProcId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (oldProcId == null && (oldProcId = this.nonceKeysToProcIdsMap.putIfAbsent(nonceKey, newProcId = this.nextProcId())) == null) {
            return -1L;
        }
        boolean isTraceEnabled = LOG.isTraceEnabled();
        while (this.isRunning() && !this.procedures.containsKey(oldProcId) && !this.completed.containsKey(oldProcId) && this.nonceKeysToProcIdsMap.containsKey(nonceKey)) {
            if (isTraceEnabled) {
                LOG.trace((Object)("waiting for procId=" + oldProcId + " to be submitted"));
            }
            Threads.sleep((long)100L);
        }
        return oldProcId;
    }

    public void unregisterNonceIfProcedureWasNotSubmitted(NonceKey nonceKey) {
        if (nonceKey == null) {
            return;
        }
        Long procId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (procId == null) {
            return;
        }
        if (!this.procedures.containsKey(procId) && !this.completed.containsKey(procId)) {
            this.nonceKeysToProcIdsMap.remove(nonceKey);
        }
    }

    public void setFailureResultForNonce(NonceKey nonceKey, String procName, User procOwner, IOException exception) {
        if (nonceKey == null) {
            return;
        }
        Long procId = this.nonceKeysToProcIdsMap.get(nonceKey);
        if (procId == null || this.completed.containsKey(procId)) {
            return;
        }
        long currentTime = EnvironmentEdgeManager.currentTime();
        ProcedureInfo result = new ProcedureInfo(procId.longValue(), procName, procOwner != null ? procOwner.getShortName() : null, ProcedureProtos.ProcedureState.ROLLEDBACK, -1L, nonceKey, ForeignExceptionUtil.toProtoForeignException((String)"ProcedureExecutor", (Throwable)exception), currentTime, currentTime, null);
        this.completed.putIfAbsent(procId, result);
    }

    public long submitProcedure(Procedure proc) {
        return this.submitProcedure(proc, null);
    }

    @SuppressWarnings(value={"NP_NULL_ON_SOME_PATH"}, justification="FindBugs is blind to the check-for-null")
    public long submitProcedure(Procedure proc, NonceKey nonceKey) {
        Long currentProcId;
        Preconditions.checkArgument((proc.getState() == ProcedureProtos.ProcedureState.INITIALIZING ? 1 : 0) != 0);
        Preconditions.checkArgument((boolean)this.isRunning(), (Object)"executor not running");
        Preconditions.checkArgument((this.lastProcId.get() >= 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((!proc.hasParent() ? 1 : 0) != 0, (String)"unexpected parent", (Object[])new Object[]{proc});
        if (nonceKey != null) {
            currentProcId = this.nonceKeysToProcIdsMap.get(nonceKey);
            Preconditions.checkArgument((currentProcId != null ? 1 : 0) != 0, (Object)("expected nonceKey=" + nonceKey + " to be reserved, use registerNonce()"));
        } else {
            currentProcId = this.nextProcId();
        }
        proc.setNonceKey(nonceKey);
        proc.setProcId(currentProcId);
        this.store.insert(proc, null);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Procedure " + proc + " added to the store."));
        }
        RootProcedureState stack = new RootProcedureState();
        this.rollbackStack.put(currentProcId, stack);
        assert (!this.procedures.containsKey(currentProcId));
        this.procedures.put(currentProcId, proc);
        this.sendProcedureAddedNotification(currentProcId);
        this.runnables.addBack(proc);
        return currentProcId;
    }

    public ProcedureInfo getResult(long procId) {
        return this.completed.get(procId);
    }

    public boolean isFinished(long procId) {
        return this.completed.containsKey(procId);
    }

    public boolean isStarted(long procId) {
        Procedure proc = this.procedures.get(procId);
        if (proc == null) {
            return this.completed.get(procId) != null;
        }
        return proc.wasExecuted();
    }

    public void removeResult(long procId) {
        ProcedureInfo result = this.completed.get(procId);
        if (result == null) {
            assert (!this.procedures.containsKey(procId)) : "procId=" + procId + " is still running";
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Procedure procId=" + procId + " already removed by the cleaner."));
            }
            return;
        }
        result.setClientAckTime(EnvironmentEdgeManager.currentTime());
    }

    public boolean abort(long procId) {
        return this.abort(procId, true);
    }

    public boolean abort(long procId, boolean mayInterruptIfRunning) {
        Procedure proc = this.procedures.get(procId);
        if (proc != null) {
            if (!mayInterruptIfRunning && proc.wasExecuted()) {
                return false;
            }
            return proc.abort(this.getEnvironment());
        }
        return false;
    }

    public boolean isProcedureOwner(long procId, User user) {
        if (user == null) {
            return false;
        }
        Procedure proc = this.procedures.get(procId);
        if (proc != null) {
            return proc.getOwner().equals(user.getShortName());
        }
        ProcedureInfo procInfo = this.completed.get(procId);
        if (procInfo == null) {
            return false;
        }
        return ProcedureInfo.isProcedureOwner((ProcedureInfo)procInfo, (User)user);
    }

    public Map<Long, ProcedureInfo> getResults() {
        return Collections.unmodifiableMap(this.completed);
    }

    public Procedure getProcedure(long procId) {
        return this.procedures.get(procId);
    }

    protected ProcedureRunnableSet getRunnableSet() {
        return this.runnables;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void execLoop() {
        while (this.isRunning()) {
            Procedure proc = this.runnables.poll();
            if (proc == null) continue;
            try {
                this.activeExecutorCount.incrementAndGet();
                this.execLoop(proc);
            }
            finally {
                this.activeExecutorCount.decrementAndGet();
            }
        }
    }

    private void execLoop(Procedure proc) {
        Long rootProcId;
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Trying to start the execution of " + proc));
        }
        if ((rootProcId = this.getRootProcedureId(proc)) == null) {
            this.executeRollback(proc);
            return;
        }
        RootProcedureState procStack = this.rollbackStack.get(rootProcId);
        if (procStack == null) {
            return;
        }
        do {
            if (!procStack.acquire(proc)) {
                if (procStack.setRollback()) {
                    if (this.executeRollback(rootProcId, procStack)) break;
                    procStack.unsetRollback();
                    this.runnables.yield(proc);
                    break;
                }
                if (proc.wasExecuted() || this.executeRollback(proc)) break;
                this.runnables.yield(proc);
                break;
            }
            assert (proc.getState() == ProcedureProtos.ProcedureState.RUNNABLE);
            if (proc.acquireLock(this.getEnvironment())) {
                this.execProcedure(procStack, proc);
                proc.releaseLock(this.getEnvironment());
            } else {
                this.runnables.yield(proc);
            }
            procStack.release(proc);
            if (this.testing != null && !this.isRunning()) break;
            if (!proc.isSuccess()) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Procedure completed in " + StringUtils.humanTimeDiff(proc.elapsedTime()) + ": " + proc));
            }
            if (proc.getProcId() != rootProcId.longValue()) break;
            this.procedureFinished(proc);
            break;
        } while (procStack.isFailed());
    }

    private void timeoutLoop() {
        while (this.isRunning()) {
            Procedure proc = this.waitingTimeout.poll();
            if (proc == null) continue;
            if (proc.getTimeRemaining() > 100L) {
                this.waitingTimeout.add(proc);
                continue;
            }
            if (proc instanceof CompletedProcedureCleaner) {
                try {
                    ((CompletedProcedureCleaner)proc).periodicExecute(this.getEnvironment());
                }
                catch (Throwable e) {
                    LOG.error((Object)("Ignoring CompletedProcedureCleaner exception: " + e.getMessage()), e);
                }
                proc.setStartTime(EnvironmentEdgeManager.currentTime());
                this.waitingTimeout.add(proc);
                continue;
            }
            if (!proc.setTimeoutFailure()) continue;
            long rootProcId = Procedure.getRootProcedureId(this.procedures, proc);
            RootProcedureState procStack = this.rollbackStack.get(rootProcId);
            procStack.abort();
            this.store.update(proc);
            this.runnables.addFront(proc);
        }
    }

    private boolean executeRollback(long rootProcId, RootProcedureState procStack) {
        Procedure rootProc = this.procedures.get(rootProcId);
        RemoteProcedureException exception = rootProc.getException();
        if (exception == null) {
            exception = procStack.getException();
            rootProc.setFailure(exception);
            this.store.update(rootProc);
        }
        List<Procedure> subprocStack = procStack.getSubprocedures();
        assert (subprocStack != null) : "Called rollback with no steps executed rootProc=" + rootProc;
        int stackTail = subprocStack.size();
        boolean reuseLock = false;
        while (stackTail-- > 0) {
            Procedure proc = subprocStack.get(stackTail);
            if (!reuseLock && !proc.acquireLock(this.getEnvironment())) {
                return false;
            }
            boolean abortRollback = !this.executeRollback(proc);
            boolean bl = reuseLock = stackTail > 0 && subprocStack.get(stackTail - 1) == proc && !(abortRollback |= !this.isRunning() || !this.store.isRunning());
            if (!reuseLock) {
                proc.releaseLock(this.getEnvironment());
            }
            if (abortRollback) {
                return false;
            }
            subprocStack.remove(stackTail);
            if (!proc.isYieldAfterExecutionStep(this.getEnvironment())) continue;
            return false;
        }
        LOG.info((Object)("Rolledback procedure " + rootProc + " exec-time=" + StringUtils.humanTimeDiff(rootProc.elapsedTime()) + " exception=" + exception.getMessage()));
        this.procedureFinished(rootProc);
        return true;
    }

    private boolean executeRollback(Procedure proc) {
        try {
            proc.doRollback(this.getEnvironment());
        }
        catch (IOException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("rollback attempt failed for " + proc), (Throwable)e);
            }
            return false;
        }
        catch (InterruptedException e) {
            this.handleInterruptedException(proc, e);
            return false;
        }
        catch (Throwable e) {
            LOG.fatal((Object)("CODE-BUG: Uncatched runtime exception for procedure: " + proc), e);
        }
        if (this.testing != null && this.testing.shouldKillBeforeStoreUpdate()) {
            LOG.debug((Object)"TESTING: Kill before store update");
            this.stop();
            return false;
        }
        if (proc.removeStackIndex()) {
            proc.setState(ProcedureProtos.ProcedureState.ROLLEDBACK);
            if (proc.hasParent()) {
                this.store.delete(proc.getProcId());
                this.procedures.remove(proc.getProcId());
            } else {
                this.store.update(proc);
            }
        } else {
            this.store.update(proc);
        }
        return true;
    }

    private void execProcedure(RootProcedureState procStack, Procedure procedure) {
        Object subproc;
        Preconditions.checkArgument((procedure.getState() == ProcedureProtos.ProcedureState.RUNNABLE ? 1 : 0) != 0);
        boolean reExecute = false;
        Object[] subprocs = null;
        do {
            reExecute = false;
            try {
                subprocs = procedure.doExecute(this.getEnvironment());
                if (subprocs != null && subprocs.length == 0) {
                    subprocs = null;
                }
            }
            catch (ProcedureYieldException e) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Yield procedure: " + procedure + ": " + e.getMessage()));
                }
                this.runnables.yield(procedure);
                return;
            }
            catch (InterruptedException e) {
                this.handleInterruptedException(procedure, e);
                this.runnables.yield(procedure);
                return;
            }
            catch (Throwable e) {
                String msg = "CODE-BUG: Uncatched runtime exception for procedure: " + procedure;
                LOG.error((Object)msg, e);
                procedure.setFailure(new RemoteProcedureException(msg, e));
            }
            if (!procedure.isFailed()) {
                if (subprocs != null) {
                    if (subprocs.length == 1 && subprocs[0] == procedure) {
                        subprocs = null;
                        reExecute = true;
                    } else {
                        for (int i = 0; i < subprocs.length; ++i) {
                            subproc = subprocs[i];
                            if (subproc == null) {
                                String msg = "subproc[" + i + "] is null, aborting the procedure";
                                procedure.setFailure(new RemoteProcedureException(msg, (Throwable)new IllegalArgumentIOException(msg)));
                                subprocs = null;
                                break;
                            }
                            assert (((Procedure)subproc).getState() == ProcedureProtos.ProcedureState.INITIALIZING);
                            ((Procedure)subproc).setParentProcId(procedure.getProcId());
                            ((Procedure)subproc).setProcId(this.nextProcId());
                        }
                        if (!procedure.isFailed()) {
                            procedure.setChildrenLatch(subprocs.length);
                            switch (procedure.getState()) {
                                case RUNNABLE: {
                                    procedure.setState(ProcedureProtos.ProcedureState.WAITING);
                                    break;
                                }
                                case WAITING_TIMEOUT: {
                                    this.waitingTimeout.add(procedure);
                                    break;
                                }
                                default: {
                                    break;
                                }
                            }
                        }
                    }
                } else if (procedure.getState() == ProcedureProtos.ProcedureState.WAITING_TIMEOUT) {
                    this.waitingTimeout.add(procedure);
                } else {
                    procedure.setState(ProcedureProtos.ProcedureState.FINISHED);
                }
            }
            procStack.addRollbackStep(procedure);
            if (this.testing != null && this.testing.shouldKillBeforeStoreUpdate()) {
                LOG.debug((Object)"TESTING: Kill before store update");
                this.stop();
                return;
            }
            if (subprocs != null && !procedure.isFailed()) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Store add " + procedure + " children " + Arrays.toString(subprocs)));
                }
                this.store.insert(procedure, (Procedure[])subprocs);
            } else {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Store update " + procedure));
                }
                this.store.update(procedure);
            }
            if (!this.store.isRunning()) {
                return;
            }
            if (procedure.getState() == ProcedureProtos.ProcedureState.RUNNABLE && procedure.isYieldAfterExecutionStep(this.getEnvironment())) {
                this.runnables.yield(procedure);
                return;
            }
            assert (reExecute && subprocs == null || !reExecute);
        } while (reExecute);
        if (subprocs != null && !procedure.isFailed()) {
            for (int i = 0; i < subprocs.length; ++i) {
                subproc = subprocs[i];
                assert (!this.procedures.containsKey(((Procedure)subproc).getProcId()));
                this.procedures.put(((Procedure)subproc).getProcId(), (Procedure)subproc);
                this.runnables.addFront((Procedure)subproc);
            }
        }
        if (procedure.isFinished() && procedure.hasParent()) {
            Procedure parent = this.procedures.get(procedure.getParentProcId());
            if (parent == null) {
                assert (procStack.isRollingback());
                return;
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)(parent + " child is done: " + procedure));
            }
            if (parent.childrenCountDown() && parent.getState() == ProcedureProtos.ProcedureState.WAITING) {
                parent.setState(ProcedureProtos.ProcedureState.RUNNABLE);
                this.store.update(parent);
                this.runnables.addFront(parent);
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)(parent + " all the children finished their work, resume."));
                }
                return;
            }
        }
    }

    private void handleInterruptedException(Procedure proc, InterruptedException e) {
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("got an interrupt during " + proc + ". suspend and retry it later."), (Throwable)e);
        }
    }

    private void sendProcedureLoadedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureLoaded(procId);
                }
                catch (Throwable e) {
                    LOG.error((Object)("The listener " + listener + " had an error: " + e.getMessage()), e);
                }
            }
        }
    }

    private void sendProcedureAddedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureAdded(procId);
                }
                catch (Throwable e) {
                    LOG.error((Object)("The listener " + listener + " had an error: " + e.getMessage()), e);
                }
            }
        }
    }

    private void sendProcedureFinishedNotification(long procId) {
        if (!this.listeners.isEmpty()) {
            for (ProcedureExecutorListener listener : this.listeners) {
                try {
                    listener.procedureFinished(procId);
                }
                catch (Throwable e) {
                    LOG.error((Object)("The listener " + listener + " had an error: " + e.getMessage()), e);
                }
            }
        }
    }

    private long nextProcId() {
        long procId = this.lastProcId.incrementAndGet();
        if (procId < 0L) {
            while (!this.lastProcId.compareAndSet(procId, 0L) && (procId = this.lastProcId.get()) < 0L) {
            }
            while (this.procedures.containsKey(procId)) {
                procId = this.lastProcId.incrementAndGet();
            }
        }
        return procId;
    }

    private Long getRootProcedureId(Procedure proc) {
        return Procedure.getRootProcedureId(this.procedures, proc);
    }

    private void procedureFinished(Procedure proc) {
        try {
            proc.completionCleanup(this.getEnvironment());
        }
        catch (Throwable e) {
            LOG.error((Object)("CODE-BUG: uncatched runtime exception for procedure: " + proc), e);
        }
        ProcedureInfo procInfo = Procedure.createProcedureInfo(proc, proc.getNonceKey());
        if (!proc.shouldWaitClientAck(this.getEnvironment())) {
            procInfo.setClientAckTime(0L);
        }
        this.completed.put(procInfo.getProcId(), procInfo);
        this.rollbackStack.remove(proc.getProcId());
        this.procedures.remove(proc.getProcId());
        try {
            this.runnables.completionCleanup(proc);
        }
        catch (Throwable e) {
            LOG.error((Object)("CODE-BUG: uncatched runtime exception for runnableSet: " + this.runnables), e);
        }
        this.sendProcedureFinishedNotification(proc.getProcId());
    }

    public Pair<ProcedureInfo, Procedure> getResultOrProcedure(long procId) {
        ProcedureInfo result = this.completed.get(procId);
        Procedure proc = null;
        if (result == null && (proc = this.procedures.get(procId)) == null) {
            result = this.completed.get(procId);
        }
        return new Pair((Object)result, proc);
    }

    private static class CompletedProcedureCleaner<TEnvironment>
    extends Procedure<TEnvironment> {
        private static final Log LOG = LogFactory.getLog(CompletedProcedureCleaner.class);
        private static final String CLEANER_INTERVAL_CONF_KEY = "hbase.procedure.cleaner.interval";
        private static final int DEFAULT_CLEANER_INTERVAL = 30000;
        private static final String EVICT_TTL_CONF_KEY = "hbase.procedure.cleaner.evict.ttl";
        private static final int DEFAULT_EVICT_TTL = 900000;
        private static final String EVICT_ACKED_TTL_CONF_KEY = "hbase.procedure.cleaner.acked.evict.ttl";
        private static final int DEFAULT_ACKED_EVICT_TTL = 300000;
        private final Map<Long, ProcedureInfo> completed;
        private final Map<NonceKey, Long> nonceKeysToProcIdsMap;
        private final ProcedureStore store;
        private final Configuration conf;

        public CompletedProcedureCleaner(Configuration conf, ProcedureStore store, Map<Long, ProcedureInfo> completedMap, Map<NonceKey, Long> nonceKeysToProcIdsMap) {
            this.setTimeout(conf.getInt(CLEANER_INTERVAL_CONF_KEY, 30000));
            this.completed = completedMap;
            this.nonceKeysToProcIdsMap = nonceKeysToProcIdsMap;
            this.store = store;
            this.conf = conf;
        }

        public void periodicExecute(TEnvironment env) {
            if (this.completed.isEmpty()) {
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)"No completed procedures to cleanup.");
                }
                return;
            }
            long evictTtl = this.conf.getInt(EVICT_TTL_CONF_KEY, 900000);
            long evictAckTtl = this.conf.getInt(EVICT_ACKED_TTL_CONF_KEY, 300000);
            long now = EnvironmentEdgeManager.currentTime();
            Iterator<Map.Entry<Long, ProcedureInfo>> it = this.completed.entrySet().iterator();
            boolean isDebugEnabled = LOG.isDebugEnabled();
            while (it.hasNext() && this.store.isRunning()) {
                Map.Entry<Long, ProcedureInfo> entry = it.next();
                ProcedureInfo procInfo = entry.getValue();
                if ((!procInfo.hasClientAckTime() || now - procInfo.getClientAckTime() < evictAckTtl) && now - procInfo.getLastUpdate() < evictTtl) continue;
                if (isDebugEnabled) {
                    LOG.debug((Object)("Evict completed procedure: " + procInfo));
                }
                this.store.delete(entry.getKey());
                it.remove();
                NonceKey nonceKey = procInfo.getNonceKey();
                if (nonceKey == null) continue;
                this.nonceKeysToProcIdsMap.remove(nonceKey);
            }
        }

        @Override
        protected Procedure[] execute(TEnvironment env) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected void rollback(TEnvironment env) {
            throw new UnsupportedOperationException();
        }

        @Override
        protected boolean abort(TEnvironment env) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void serializeStateData(OutputStream stream) {
            throw new UnsupportedOperationException();
        }

        @Override
        public void deserializeStateData(InputStream stream) {
            throw new UnsupportedOperationException();
        }
    }

    private static class ProcedureTimeoutRetriever
    implements TimeoutBlockingQueue.TimeoutRetriever<Procedure> {
        private ProcedureTimeoutRetriever() {
        }

        @Override
        public long getTimeout(Procedure proc) {
            return proc.getTimeRemaining();
        }

        @Override
        public TimeUnit getTimeUnit(Procedure proc) {
            return TimeUnit.MILLISECONDS;
        }
    }

    public static interface ProcedureExecutorListener {
        public void procedureLoaded(long var1);

        public void procedureAdded(long var1);

        public void procedureFinished(long var1);
    }

    public static class Testing {
        protected boolean killBeforeStoreUpdate = false;
        protected boolean toggleKillBeforeStoreUpdate = false;

        protected boolean shouldKillBeforeStoreUpdate() {
            boolean kill = this.killBeforeStoreUpdate;
            if (this.toggleKillBeforeStoreUpdate) {
                this.killBeforeStoreUpdate = !kill;
                LOG.warn((Object)("Toggle Kill before store update to: " + this.killBeforeStoreUpdate));
            }
            return kill;
        }
    }
}

