/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.tools.offlineImageViewer;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.LimitInputStream;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.permission.PermissionStatus;
import org.apache.hadoop.hdfs.protocol.proto.HdfsProtos;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormatPBINode;
import org.apache.hadoop.hdfs.server.namenode.FSImageFormatProtobuf;
import org.apache.hadoop.hdfs.server.namenode.FSImageUtil;
import org.apache.hadoop.hdfs.server.namenode.FsImageProto;
import org.apache.hadoop.hdfs.tools.offlineImageViewer.FSImageHandler;
import org.apache.hadoop.io.IOUtils;
import org.codehaus.jackson.map.ObjectMapper;

class FSImageLoader {
    public static final Log LOG = LogFactory.getLog(FSImageHandler.class);
    private static String[] stringTable;
    private static Map<Long, FsImageProto.INodeSection.INode> inodes;
    private static Map<Long, long[]> dirmap;
    private static List<FsImageProto.INodeReferenceSection.INodeReference> refList;

    private FSImageLoader() {
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static FSImageLoader load(String inputFile) throws IOException {
        Configuration conf = new Configuration();
        RandomAccessFile file = new RandomAccessFile(inputFile, "r");
        if (!FSImageUtil.checkFileFormat(file)) {
            throw new IOException("Unrecognized FSImage");
        }
        FsImageProto.FileSummary summary = FSImageUtil.loadSummary(file);
        FileInputStream fin = null;
        try {
            fin = new FileInputStream(file.getFD());
            ArrayList sections = Lists.newArrayList(summary.getSectionsList());
            Collections.sort(sections, new Comparator<FsImageProto.FileSummary.Section>(){

                @Override
                public int compare(FsImageProto.FileSummary.Section s1, FsImageProto.FileSummary.Section s2) {
                    FSImageFormatProtobuf.SectionName n1 = FSImageFormatProtobuf.SectionName.fromString(s1.getName());
                    FSImageFormatProtobuf.SectionName n2 = FSImageFormatProtobuf.SectionName.fromString(s2.getName());
                    if (n1 == null) {
                        return n2 == null ? 0 : -1;
                    }
                    if (n2 == null) {
                        return -1;
                    }
                    return n1.ordinal() - n2.ordinal();
                }
            });
            for (FsImageProto.FileSummary.Section s : sections) {
                fin.getChannel().position(s.getOffset());
                InputStream is = FSImageUtil.wrapInputStreamForCompression(conf, summary.getCodec(), new BufferedInputStream((InputStream)new LimitInputStream((InputStream)fin, s.getLength())));
                switch (FSImageFormatProtobuf.SectionName.fromString(s.getName())) {
                    case STRING_TABLE: {
                        FSImageLoader.loadStringTable(is);
                        break;
                    }
                    case INODE: {
                        FSImageLoader.loadINodeSection(is);
                        break;
                    }
                    case INODE_REFERENCE: {
                        FSImageLoader.loadINodeReferenceSection(is);
                        break;
                    }
                    case INODE_DIR: {
                        FSImageLoader.loadINodeDirectorySection(is);
                        break;
                    }
                }
            }
        }
        catch (Throwable throwable) {
            IOUtils.cleanup(null, (Closeable[])new Closeable[]{fin});
            throw throwable;
        }
        IOUtils.cleanup(null, (Closeable[])new Closeable[]{fin});
        return new FSImageLoader();
    }

    private static void loadINodeDirectorySection(InputStream in) throws IOException {
        FsImageProto.INodeDirectorySection.DirEntry e;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)"Loading directory section");
        }
        while ((e = FsImageProto.INodeDirectorySection.DirEntry.parseDelimitedFrom(in)) != null) {
            int i;
            long[] l = new long[e.getChildrenCount() + e.getRefChildrenCount()];
            for (i = 0; i < e.getChildrenCount(); ++i) {
                l[i] = e.getChildren(i);
            }
            for (i = e.getChildrenCount(); i < l.length; ++i) {
                int refId = e.getRefChildren(i - e.getChildrenCount());
                l[i] = refList.get(refId).getReferredId();
            }
            dirmap.put(e.getParent(), l);
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug((Object)("Loaded directory (parent " + e.getParent() + ") with " + e.getChildrenCount() + " children and " + e.getRefChildrenCount() + " reference children"));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Loaded " + dirmap.size() + " directories"));
        }
    }

    private static void loadINodeReferenceSection(InputStream in) throws IOException {
        FsImageProto.INodeReferenceSection.INodeReference e;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)"Loading inode reference section");
        }
        while ((e = FsImageProto.INodeReferenceSection.INodeReference.parseDelimitedFrom(in)) != null) {
            refList.add(e);
            if (!LOG.isTraceEnabled()) continue;
            LOG.trace((Object)("Loaded inode reference named '" + e.getName() + "' referring to id " + e.getReferredId() + ""));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Loaded " + refList.size() + " inode references"));
        }
    }

    private static void loadINodeSection(InputStream in) throws IOException {
        FsImageProto.INodeSection s = FsImageProto.INodeSection.parseDelimitedFrom(in);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Found " + s.getNumInodes() + " inodes in inode section"));
        }
        int i = 0;
        while ((long)i < s.getNumInodes()) {
            FsImageProto.INodeSection.INode p = FsImageProto.INodeSection.INode.parseDelimitedFrom(in);
            inodes.put(p.getId(), p);
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Loaded inode id " + p.getId() + " type " + (Object)((Object)p.getType()) + " name '" + p.getName().toStringUtf8() + "'"));
            }
            ++i;
        }
    }

    private static void loadStringTable(InputStream in) throws IOException {
        FsImageProto.StringTableSection s = FsImageProto.StringTableSection.parseDelimitedFrom(in);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Found " + s.getNumEntry() + " strings in string section"));
        }
        stringTable = new String[s.getNumEntry() + 1];
        for (int i = 0; i < s.getNumEntry(); ++i) {
            FsImageProto.StringTableSection.Entry e = FsImageProto.StringTableSection.Entry.parseDelimitedFrom(in);
            FSImageLoader.stringTable[e.getId()] = e.getStr();
            if (!LOG.isTraceEnabled()) continue;
            LOG.trace((Object)("Loaded string " + e.getStr()));
        }
    }

    String getFileStatus(String path) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        FsImageProto.INodeSection.INode inode = inodes.get(this.getINodeId(path));
        return "{\"FileStatus\":\n" + mapper.writeValueAsString(this.getFileStatus(inode, false)) + "\n}\n";
    }

    String listStatus(String path) throws IOException {
        StringBuilder sb = new StringBuilder();
        ObjectMapper mapper = new ObjectMapper();
        List<Map<String, Object>> fileStatusList = this.getFileStatusList(path);
        sb.append("{\"FileStatuses\":{\"FileStatus\":[\n");
        int i = 0;
        for (Map<String, Object> fileStatusMap : fileStatusList) {
            if (i++ != 0) {
                sb.append(',');
            }
            sb.append(mapper.writeValueAsString(fileStatusMap));
        }
        sb.append("\n]}}\n");
        return sb.toString();
    }

    private List<Map<String, Object>> getFileStatusList(String path) {
        ArrayList<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
        long id = this.getINodeId(path);
        FsImageProto.INodeSection.INode inode = inodes.get(id);
        if (inode.getType() == FsImageProto.INodeSection.INode.Type.DIRECTORY) {
            long[] children;
            if (!dirmap.containsKey(id)) {
                return list;
            }
            for (long cid : children = dirmap.get(id)) {
                list.add(this.getFileStatus(inodes.get(cid), true));
            }
        } else {
            list.add(this.getFileStatus(inode, false));
        }
        return list;
    }

    String getAclStatus(String path) throws IOException {
        StringBuilder sb = new StringBuilder();
        List<AclEntry> aclEntryList = this.getAclEntryList(path);
        PermissionStatus p = this.getPermissionStatus(path);
        sb.append("{\"AclStatus\":{\"entries\":[");
        int i = 0;
        for (AclEntry aclEntry : aclEntryList) {
            if (i++ != 0) {
                sb.append(',');
            }
            sb.append('\"');
            sb.append(aclEntry.toString());
            sb.append('\"');
        }
        sb.append("],\"group\": \"");
        sb.append(p.getGroupName());
        sb.append("\",\"owner\": \"");
        sb.append(p.getUserName());
        sb.append("\",\"stickyBit\": ");
        sb.append(p.getPermission().getStickyBit());
        sb.append("}}\n");
        return sb.toString();
    }

    private List<AclEntry> getAclEntryList(String path) {
        long id = this.getINodeId(path);
        FsImageProto.INodeSection.INode inode = inodes.get(id);
        switch (inode.getType()) {
            case FILE: {
                FsImageProto.INodeSection.INodeFile f = inode.getFile();
                return FSImageFormatPBINode.Loader.loadAclEntries(f.getAcl(), stringTable);
            }
            case DIRECTORY: {
                FsImageProto.INodeSection.INodeDirectory d = inode.getDirectory();
                return FSImageFormatPBINode.Loader.loadAclEntries(d.getAcl(), stringTable);
            }
        }
        return new ArrayList<AclEntry>();
    }

    private PermissionStatus getPermissionStatus(String path) {
        long id = this.getINodeId(path);
        FsImageProto.INodeSection.INode inode = inodes.get(id);
        switch (inode.getType()) {
            case FILE: {
                FsImageProto.INodeSection.INodeFile f = inode.getFile();
                return FSImageFormatPBINode.Loader.loadPermission(f.getPermission(), stringTable);
            }
            case DIRECTORY: {
                FsImageProto.INodeSection.INodeDirectory d = inode.getDirectory();
                return FSImageFormatPBINode.Loader.loadPermission(d.getPermission(), stringTable);
            }
            case SYMLINK: {
                FsImageProto.INodeSection.INodeSymlink s = inode.getSymlink();
                return FSImageFormatPBINode.Loader.loadPermission(s.getPermission(), stringTable);
            }
        }
        return null;
    }

    private long getINodeId(String strPath) {
        if (strPath.equals("/")) {
            return 16385L;
        }
        String[] nameList = strPath.split("/");
        Preconditions.checkArgument((nameList.length > 1 ? 1 : 0) != 0, (Object)("Illegal path: " + strPath));
        long id = 16385L;
        for (int i = 1; i < nameList.length; ++i) {
            long[] children = dirmap.get(id);
            Preconditions.checkNotNull((Object)children, (Object)("File: " + strPath + " is not found in the fsimage."));
            String cName = nameList[i];
            boolean findChildren = false;
            for (long cid : children) {
                if (!cName.equals(inodes.get(cid).getName().toStringUtf8())) continue;
                id = cid;
                findChildren = true;
                break;
            }
            Preconditions.checkArgument((boolean)findChildren, (Object)("File: " + strPath + " is not found in the fsimage."));
        }
        return id;
    }

    private Map<String, Object> getFileStatus(FsImageProto.INodeSection.INode inode, boolean printSuffix) {
        HashMap map = Maps.newHashMap();
        switch (inode.getType()) {
            case FILE: {
                FsImageProto.INodeSection.INodeFile f = inode.getFile();
                PermissionStatus p = FSImageFormatPBINode.Loader.loadPermission(f.getPermission(), stringTable);
                map.put("accessTime", f.getAccessTime());
                map.put("blockSize", f.getPreferredBlockSize());
                map.put("group", p.getGroupName());
                map.put("length", this.getFileSize(f));
                map.put("modificationTime", f.getModificationTime());
                map.put("owner", p.getUserName());
                map.put("pathSuffix", printSuffix ? inode.getName().toStringUtf8() : "");
                map.put("permission", this.toString(p.getPermission()));
                map.put("replication", f.getReplication());
                map.put("type", inode.getType());
                map.put("fileId", inode.getId());
                map.put("childrenNum", 0);
                return map;
            }
            case DIRECTORY: {
                FsImageProto.INodeSection.INodeDirectory d = inode.getDirectory();
                PermissionStatus p = FSImageFormatPBINode.Loader.loadPermission(d.getPermission(), stringTable);
                map.put("accessTime", 0);
                map.put("blockSize", 0);
                map.put("group", p.getGroupName());
                map.put("length", 0);
                map.put("modificationTime", d.getModificationTime());
                map.put("owner", p.getUserName());
                map.put("pathSuffix", printSuffix ? inode.getName().toStringUtf8() : "");
                map.put("permission", this.toString(p.getPermission()));
                map.put("replication", 0);
                map.put("type", inode.getType());
                map.put("fileId", inode.getId());
                map.put("childrenNum", dirmap.containsKey(inode.getId()) ? dirmap.get(inode.getId()).length : 0);
                return map;
            }
            case SYMLINK: {
                FsImageProto.INodeSection.INodeSymlink d = inode.getSymlink();
                PermissionStatus p = FSImageFormatPBINode.Loader.loadPermission(d.getPermission(), stringTable);
                map.put("accessTime", d.getAccessTime());
                map.put("blockSize", 0);
                map.put("group", p.getGroupName());
                map.put("length", 0);
                map.put("modificationTime", d.getModificationTime());
                map.put("owner", p.getUserName());
                map.put("pathSuffix", printSuffix ? inode.getName().toStringUtf8() : "");
                map.put("permission", this.toString(p.getPermission()));
                map.put("replication", 0);
                map.put("type", inode.getType());
                map.put("symlink", d.getTarget().toStringUtf8());
                map.put("fileId", inode.getId());
                map.put("childrenNum", 0);
                return map;
            }
        }
        return null;
    }

    private long getFileSize(FsImageProto.INodeSection.INodeFile f) {
        long size = 0L;
        for (HdfsProtos.BlockProto p : f.getBlocksList()) {
            size += p.getNumBytes();
        }
        return size;
    }

    private String toString(FsPermission permission) {
        return String.format("%o", permission.toShort());
    }

    static {
        inodes = Maps.newHashMap();
        dirmap = Maps.newHashMap();
        refList = Lists.newArrayList();
    }
}

