/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite.internal.processors.cache.persistence.file;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.ignite.IgniteCheckedException;
import org.apache.ignite.IgniteSystemProperties;
import org.apache.ignite.configuration.DataStorageConfiguration;
import org.apache.ignite.internal.pagemem.PageIdUtils;
import org.apache.ignite.internal.pagemem.store.PageStore;
import org.apache.ignite.internal.processors.cache.persistence.AllocatedPageTracker;
import org.apache.ignite.internal.processors.cache.persistence.file.FileIO;
import org.apache.ignite.internal.processors.cache.persistence.file.FileIOFactory;
import org.apache.ignite.internal.processors.cache.persistence.file.PersistentStorageIOException;
import org.apache.ignite.internal.processors.cache.persistence.tree.io.PageIO;
import org.apache.ignite.internal.processors.cache.persistence.wal.crc.IgniteDataIntegrityViolationException;
import org.apache.ignite.internal.processors.cache.persistence.wal.crc.PureJavaCrc32;
import org.apache.ignite.internal.util.typedef.internal.U;

public class FilePageStore
implements PageStore {
    private static final long SIGNATURE = -1037300167331204936L;
    public static final int VERSION = 1;
    public static final int HEADER_SIZE = 17;
    private final File cfgFile;
    private final byte type;
    protected final DataStorageConfiguration dbCfg;
    private final FileIOFactory ioFactory;
    private FileIO fileIO;
    private final AtomicLong allocated;
    private final AllocatedPageTracker allocatedTracker;
    private final int pageSize;
    private volatile boolean inited;
    private volatile boolean recover;
    private volatile int tag;
    private boolean skipCrc = IgniteSystemProperties.getBoolean("IGNITE_PDS_SKIP_CRC", false);
    private final ReadWriteLock lock = new ReentrantReadWriteLock();

    public FilePageStore(byte type, File file, FileIOFactory factory, DataStorageConfiguration cfg, AllocatedPageTracker allocatedTracker) {
        this.type = type;
        this.cfgFile = file;
        this.dbCfg = cfg;
        this.ioFactory = factory;
        this.allocated = new AtomicLong();
        this.pageSize = this.dbCfg.getPageSize();
        this.allocatedTracker = allocatedTracker;
    }

    @Override
    public boolean exists() {
        return this.cfgFile.exists() && this.cfgFile.length() > (long)this.headerSize();
    }

    public int headerSize() {
        return 17;
    }

    @Override
    public int version() {
        return 1;
    }

    public ByteBuffer header(byte type, int pageSize) {
        ByteBuffer hdr = ByteBuffer.allocate(this.headerSize()).order(ByteOrder.LITTLE_ENDIAN);
        hdr.putLong(-1037300167331204936L);
        hdr.putInt(this.version());
        hdr.put(type);
        hdr.putInt(pageSize);
        hdr.rewind();
        return hdr;
    }

    private long initFile() throws IOException {
        ByteBuffer hdr = this.header(this.type, this.dbCfg.getPageSize());
        while (hdr.remaining() > 0) {
            this.fileIO.write(hdr);
        }
        return this.headerSize() + this.dbCfg.getPageSize();
    }

    private long checkFile() throws PersistentStorageIOException {
        try {
            ByteBuffer hdr = ByteBuffer.allocate(this.headerSize()).order(ByteOrder.LITTLE_ENDIAN);
            while (hdr.remaining() > 0) {
                this.fileIO.read(hdr);
            }
            hdr.rewind();
            long signature = hdr.getLong();
            if (-1037300167331204936L != signature) {
                throw new IOException("Failed to verify store file (invalid file signature) [expectedSignature=" + U.hexLong(-1037300167331204936L) + ", actualSignature=" + U.hexLong(signature) + ']');
            }
            int ver = hdr.getInt();
            if (this.version() != ver) {
                throw new IOException("Failed to verify store file (invalid file version) [expectedVersion=" + this.version() + ", fileVersion=" + ver + "]");
            }
            byte type = hdr.get();
            if (this.type != type) {
                throw new IOException("Failed to verify store file (invalid file type) [expectedFileType=" + this.type + ", actualFileType=" + type + "]");
            }
            int pageSize = hdr.getInt();
            if (this.dbCfg.getPageSize() != pageSize) {
                throw new IOException("Failed to verify store file (invalid page size) [expectedPageSize=" + this.dbCfg.getPageSize() + ", filePageSize=" + pageSize + "]");
            }
            long fileSize = this.cfgFile.length();
            if (fileSize == (long)this.headerSize()) {
                fileSize = pageSize + this.headerSize();
            }
            if ((fileSize - (long)this.headerSize()) % (long)pageSize != 0L) {
                throw new IOException("Failed to verify store file (invalid file size) [fileSize=" + U.hexLong(fileSize) + ", pageSize=" + U.hexLong(pageSize) + ']');
            }
            return fileSize;
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("File check failed", e);
        }
    }

    public void stop(boolean cleanFile) throws PersistentStorageIOException {
        this.lock.writeLock().lock();
        try {
            if (!this.inited) {
                return;
            }
            this.fileIO.force();
            this.fileIO.close();
            if (cleanFile) {
                Files.delete(this.cfgFile.toPath());
            }
        }
        catch (IOException e) {
            throw new PersistentStorageIOException(e);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public void truncate(int tag) throws PersistentStorageIOException {
        this.init();
        this.lock.writeLock().lock();
        try {
            this.tag = tag;
            this.fileIO.clear();
            this.fileIO.close();
            Files.delete(this.cfgFile.toPath());
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("Failed to delete partition file: " + this.cfgFile.getPath(), e);
        }
        finally {
            this.allocated.set(0L);
            this.allocatedTracker.updateTotalAllocatedPages(-1L * (long)this.pages());
            this.inited = false;
            this.lock.writeLock().unlock();
        }
    }

    public void beginRecover() {
        this.lock.writeLock().lock();
        try {
            this.recover = true;
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    public void finishRecover() throws PersistentStorageIOException {
        this.lock.writeLock().lock();
        try {
            if (this.inited) {
                long newSize = Math.max((long)(this.headerSize() + this.pageSize), this.fileIO.size());
                long delta = newSize - this.allocated.getAndSet(newSize);
                assert (delta % (long)this.pageSize == 0L);
                this.allocatedTracker.updateTotalAllocatedPages(delta / (long)this.pageSize);
            }
            this.recover = false;
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("Failed to finish recover", e);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public void read(long pageId, ByteBuffer pageBuf, boolean keepCrc) throws IgniteCheckedException {
        this.init();
        try {
            int curCrc32;
            int n;
            long off = this.pageOffset(pageId);
            assert (pageBuf.capacity() == this.pageSize);
            assert (pageBuf.position() == 0);
            assert (pageBuf.order() == ByteOrder.nativeOrder());
            assert (off <= this.allocated.get() - (long)this.headerSize()) : "calculatedOffset=" + off + ", allocated=" + this.allocated.get() + ", headerSize=" + this.headerSize();
            int len = this.pageSize;
            do {
                if ((n = this.fileIO.read(pageBuf, off)) < 0) {
                    pageBuf.put(new byte[pageBuf.remaining()]);
                    return;
                }
                off += (long)n;
            } while ((len -= n) > 0);
            int savedCrc32 = PageIO.getCrc(pageBuf);
            PageIO.setCrc(pageBuf, 0);
            pageBuf.position(0);
            if (!this.skipCrc && (savedCrc32 ^ (curCrc32 = PureJavaCrc32.calcCrc32(pageBuf, this.pageSize))) != 0) {
                throw new IgniteDataIntegrityViolationException("Failed to read page (CRC validation failed) [id=" + U.hexLong(pageId) + ", off=" + (off - (long)this.pageSize) + ", file=" + this.cfgFile.getAbsolutePath() + ", fileSize=" + this.fileIO.size() + ", savedCrc=" + U.hexInt(savedCrc32) + ", curCrc=" + U.hexInt(curCrc32) + ", page=" + U.toHexString(pageBuf) + "]");
            }
            assert (PageIO.getCrc(pageBuf) == 0);
            if (keepCrc) {
                PageIO.setCrc(pageBuf, savedCrc32);
            }
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("Read error", e);
        }
    }

    @Override
    public void readHeader(ByteBuffer buf) throws IgniteCheckedException {
        this.init();
        try {
            int n;
            assert (buf.remaining() == this.headerSize());
            int len = this.headerSize();
            long off = 0L;
            do {
                if ((n = this.fileIO.read(buf, off)) < 0) {
                    return;
                }
                off += (long)n;
            } while ((len -= n) > 0);
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("Read error", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void init() throws PersistentStorageIOException {
        block16: {
            if (!this.inited) {
                this.lock.writeLock().lock();
                try {
                    if (this.inited) break block16;
                    FileIO fileIO = null;
                    Throwable err = null;
                    try {
                        long newSize;
                        this.fileIO = fileIO = this.ioFactory.create(this.cfgFile, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
                        long l = newSize = this.cfgFile.length() == 0L ? this.initFile() : this.checkFile();
                        assert (this.allocated.get() == 0L);
                        long delta = newSize - (long)this.headerSize();
                        assert (delta % (long)this.pageSize == 0L);
                        this.allocatedTracker.updateTotalAllocatedPages(delta / (long)this.pageSize);
                        this.allocated.set(newSize);
                        this.inited = true;
                    }
                    catch (IOException e) {
                        err = new PersistentStorageIOException("Failed to initialize partition file: " + this.cfgFile.getName(), e);
                        throw err;
                    }
                    finally {
                        if (err != null && fileIO != null) {
                            try {
                                fileIO.close();
                            }
                            catch (IOException e) {
                                err.addSuppressed(e);
                            }
                        }
                    }
                }
                finally {
                    this.lock.writeLock().unlock();
                }
            }
        }
    }

    @Override
    public void write(long pageId, ByteBuffer pageBuf, int tag, boolean calculateCrc) throws IgniteCheckedException {
        this.init();
        this.lock.readLock().lock();
        try {
            int n;
            if (tag < this.tag) {
                return;
            }
            long off = this.pageOffset(pageId);
            assert (off >= 0L && off + (long)this.headerSize() <= this.allocated.get() || this.recover) : "off=" + U.hexLong(off) + ", allocated=" + U.hexLong(this.allocated.get()) + ", pageId=" + U.hexLong(pageId);
            assert (pageBuf.capacity() == this.pageSize);
            assert (pageBuf.position() == 0);
            assert (pageBuf.order() == ByteOrder.nativeOrder()) : "Page buffer order " + pageBuf.order() + " should be same with " + ByteOrder.nativeOrder();
            assert (PageIO.getType(pageBuf) != 0) : "Invalid state. Type is 0! pageId = " + U.hexLong(pageId);
            assert (PageIO.getVersion(pageBuf) != 0) : "Invalid state. Version is 0! pageId = " + U.hexLong(pageId);
            if (calculateCrc && !this.skipCrc) {
                assert (PageIO.getCrc(pageBuf) == 0) : U.hexLong(pageId);
                PageIO.setCrc(pageBuf, FilePageStore.calcCrc32(pageBuf, this.pageSize));
            }
            assert (this.skipCrc || PageIO.getCrc(pageBuf) != 0 || FilePageStore.calcCrc32(pageBuf, this.pageSize) == 0) : "CRC hasn't been calculated, crc=0";
            assert (pageBuf.position() == 0) : pageBuf.position();
            int len = this.pageSize;
            do {
                n = this.fileIO.write(pageBuf, off);
                off += (long)n;
            } while ((len -= n) > 0);
            PageIO.setCrc(pageBuf, 0);
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("Failed to write the page to the file store [pageId=" + pageId + ", file=" + this.cfgFile.getAbsolutePath() + ']', e);
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private static int calcCrc32(ByteBuffer pageBuf, int pageSize) {
        try {
            pageBuf.position(0);
            int n = PureJavaCrc32.calcCrc32(pageBuf, pageSize);
            return n;
        }
        finally {
            pageBuf.position(0);
        }
    }

    @Override
    public long pageOffset(long pageId) {
        return (long)PageIdUtils.pageIndex(pageId) * (long)this.pageSize + (long)this.headerSize();
    }

    @Override
    public void sync() throws IgniteCheckedException {
        this.lock.writeLock().lock();
        try {
            this.init();
            this.fileIO.force();
        }
        catch (IOException e) {
            throw new PersistentStorageIOException("Sync error", e);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    @Override
    public synchronized void ensure() throws IgniteCheckedException {
        this.init();
    }

    @Override
    public long allocatePage() throws IgniteCheckedException {
        this.init();
        long off = this.allocPage();
        return (off - (long)this.headerSize()) / (long)this.pageSize;
    }

    private long allocPage() {
        long off;
        while (!this.allocated.compareAndSet(off = this.allocated.get(), off + (long)this.pageSize)) {
        }
        this.allocatedTracker.updateTotalAllocatedPages(1L);
        return off;
    }

    @Override
    public int pages() {
        if (!this.inited) {
            return 0;
        }
        return (int)((this.allocated.get() - (long)this.headerSize()) / (long)this.pageSize);
    }
}

